<?php
session_start();
header('Content-Type: application/json'); // Set content type for JSON responses

include('db.php'); // Assumes $conn is established here

// Helper function to send JSON response and exit
function send_json_response($status, $message, $data = []) {
    echo json_encode(array_merge(['status' => $status, 'message' => $message], $data));
    if (isset($GLOBALS['conn']) && $GLOBALS['conn']) { // Close connection if open
        mysqli_close($GLOBALS['conn']);
    }
    // Close any prepared statements if they were passed or globally accessible (safer to close them in finally block)
    exit;
}

$user_id = $_SESSION['user_id'] ?? null;

if ($user_id === null) {
    send_json_response('error', 'User not logged in.');
}
// Ensure user_id is an integer if it's used as such in the DB
$user_id = (int) $user_id;

$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (json_last_error() !== JSON_ERROR_NONE) {
    send_json_response('error', 'Invalid JSON input: ' . json_last_error_msg());
}

$product_id = $data['product_id'] ?? null;
$quantity_to_add = $data['quantity'] ?? null; // Renamed for clarity as it's an addition

if ($product_id === null) {
    send_json_response('error', 'Product ID is required.');
}
if (!is_numeric($product_id) || (int)$product_id <= 0) {
    send_json_response('error', 'Product ID must be a positive integer.');
}
$product_id = (int)$product_id;

if ($quantity_to_add === null) {
    send_json_response('error', 'Quantity is required.');
}
// Allow any numeric quantity (positive, negative, or zero) as original code implicitly allowed this
if (!is_numeric($quantity_to_add)) {
    send_json_response('error', 'Quantity must be a number.');
}
$quantity_to_add = (int)$quantity_to_add;


$stmt_check = null;
$stmt_update = null;

try {
    // Check if the product already exists in the cart for the user
    $query_check = "
        SELECT Quantity
        FROM cart
        WHERE User_id = ? AND Product_id = ? AND Delete_status = 0
    ";
    $stmt_check = mysqli_prepare($conn, $query_check);
    if (!$stmt_check) {
        error_log("MySQLi prepare error (check): " . mysqli_error($conn));
        send_json_response('error', 'Database error. Please try again later. (prep_check)');
    }

    mysqli_stmt_bind_param($stmt_check, "ii", $user_id, $product_id);

    if (!mysqli_stmt_execute($stmt_check)) {
        error_log("MySQLi execute error (check): " . mysqli_stmt_error($stmt_check));
        send_json_response('error', 'Database error. Please try again later. (exec_check)');
    }

    $result_check = mysqli_stmt_get_result($stmt_check);

    if (mysqli_num_rows($result_check) > 0) {
        // Product found, update its quantity
        $query_update = "
            UPDATE cart
            SET Quantity = Quantity + ?
            WHERE User_id = ? AND Product_id = ? AND Delete_status = 0
        ";
        $stmt_update = mysqli_prepare($conn, $query_update);
        if (!$stmt_update) {
            error_log("MySQLi prepare error (update): " . mysqli_error($conn));
            send_json_response('error', 'Database error. Please try again later. (prep_update)');
        }

        mysqli_stmt_bind_param($stmt_update, "iii", $quantity_to_add, $user_id, $product_id);

        if (mysqli_stmt_execute($stmt_update)) {
            // Original code considered successful execution as success, regardless of affected rows.
            // We'll keep that behavior. For stricter checking, one might check mysqli_stmt_affected_rows.
            send_json_response('success', 'Quantity updated successfully.');
        } else {
            error_log("MySQLi execute error (update): " . mysqli_stmt_error($stmt_update));
            send_json_response('error', 'Failed to update quantity.');
        }
    } else {
        // Product not found in the cart for this user with Delete_status = 0
        send_json_response('error', 'Product not found in the cart.');
    }

} catch (Exception $e) {
    // Catch any other unexpected PHP errors
    error_log("General Error: " . $e->getMessage() . " in " . $e->getFile() . " on line " . $e->getLine());
    send_json_response('error', 'An unexpected error occurred. Please try again later.');
} finally {
    if ($stmt_check) {
        mysqli_stmt_close($stmt_check);
    }
    if ($stmt_update) {
        mysqli_stmt_close($stmt_update);
    }
    if ($conn) {
        mysqli_close($conn);
    }
}
?>